/**
 * 选项页面脚本
 */
const api = new TranslationAPI();

// DOM元素
const apiUrlInput = document.getElementById('api-url');
const defaultSourceLangSelect = document.getElementById('default-source-lang');
const defaultTargetLangSelect = document.getElementById('default-target-lang');
const useAIDefaultCheckbox = document.getElementById('use-ai-default');
const autoTranslateSelectionCheckbox = document.getElementById('auto-translate-selection');
const showTranslationPopupCheckbox = document.getElementById('show-translation-popup');
const saveBtn = document.getElementById('save-btn');
const resetBtn = document.getElementById('reset-btn');
const saveStatus = document.getElementById('save-status');

/**
 * 初始化
 */
async function init() {
    await api.init();
    await loadLanguages();
    await loadSettings();
    bindEvents();
}

/**
 * 加载语言列表
 */
async function loadLanguages() {
    try {
        const result = await api.getLanguages();
        if (result.success && result.languages) {
            const languages = result.languages;

            // 填充源语言选择器
            defaultSourceLangSelect.innerHTML = '';
            languages.forEach(lang => {
                const option = document.createElement('option');
                option.value = lang.code;
                option.textContent = lang.native_name || lang.name;
                defaultSourceLangSelect.appendChild(option);
            });

            // 填充目标语言选择器
            defaultTargetLangSelect.innerHTML = '';
            languages.filter(lang => lang.code !== 'auto').forEach(lang => {
                const option = document.createElement('option');
                option.value = lang.code;
                option.textContent = lang.native_name || lang.name;
                defaultTargetLangSelect.appendChild(option);
            });
        }
    } catch (error) {
        console.error('加载语言列表失败:', error);
    }
}

/**
 * 加载设置
 */
async function loadSettings() {
    const config = await api.getConfig();

    apiUrlInput.value = config.apiUrl || 'https://translate.worldhub.world';
    defaultSourceLangSelect.value = config.defaultSourceLang || 'auto';
    defaultTargetLangSelect.value = config.defaultTargetLang || 'zh-CN';
    useAIDefaultCheckbox.checked = config.useAI || false;

    // 加载其他设置
    return new Promise((resolve) => {
        const storage = typeof chrome !== 'undefined' && chrome.storage
            ? chrome.storage
            : browser.storage;

        storage.sync.get(['autoTranslateSelection', 'showTranslationPopup'], (items) => {
            autoTranslateSelectionCheckbox.checked = items.autoTranslateSelection !== false;
            showTranslationPopupCheckbox.checked = items.showTranslationPopup !== false;
            resolve();
        });
    });
}

/**
 * 绑定事件
 */
function bindEvents() {
    saveBtn.addEventListener('click', handleSave);
    resetBtn.addEventListener('click', handleReset);
}

/**
 * 保存设置
 */
async function handleSave() {
    const config = {
        apiUrl: apiUrlInput.value.trim() || 'https://translate.worldhub.world',
        defaultSourceLang: defaultSourceLangSelect.value,
        defaultTargetLang: defaultTargetLangSelect.value,
        useAI: useAIDefaultCheckbox.checked,
        autoTranslateSelection: autoTranslateSelectionCheckbox.checked,
        showTranslationPopup: showTranslationPopupCheckbox.checked
    };

    try {
        const storage = typeof chrome !== 'undefined' && chrome.storage
            ? chrome.storage
            : browser.storage;

        await new Promise((resolve, reject) => {
            storage.sync.set(config, () => {
                if (chrome.runtime && chrome.runtime.lastError) {
                    reject(new Error(chrome.runtime.lastError.message));
                } else {
                    resolve();
                }
            });
        });

        // 更新API的baseURL
        api.baseURL = config.apiUrl;

        showStatus('设置已保存', 'success');
    } catch (error) {
        console.error('保存设置失败:', error);
        showStatus('保存失败: ' + error.message, 'error');
    }
}

/**
 * 重置设置
 */
async function handleReset() {
    if (confirm('确定要重置所有设置吗？')) {
        apiUrlInput.value = 'https://translate.worldhub.world';
        defaultSourceLangSelect.value = 'auto';
        defaultTargetLangSelect.value = 'zh-CN';
        useAIDefaultCheckbox.checked = false;
        autoTranslateSelectionCheckbox.checked = true;
        showTranslationPopupCheckbox.checked = true;

        await handleSave();
        showStatus('设置已重置', 'success');
    }
}

/**
 * 显示状态
 */
function showStatus(message, type = 'success') {
    saveStatus.textContent = message;
    saveStatus.className = `save-status save-status-${type}`;
    saveStatus.style.display = 'block';

    setTimeout(() => {
        saveStatus.style.display = 'none';
    }, 3000);
}

// 初始化
init().catch(console.error);

