/**
 * Popup脚本
 */
const api = new TranslationAPI();

// DOM元素
const textInput = document.getElementById('text-input');
const sourceLangSelect = document.getElementById('source-lang');
const targetLangSelect = document.getElementById('target-lang');
const useAICheckbox = document.getElementById('use-ai');
const translateBtn = document.getElementById('translate-btn');
const translatePageBtn = document.getElementById('translate-page-btn');
const resultSection = document.getElementById('result-section');
const resultText = document.getElementById('result-text');
const errorSection = document.getElementById('error-section');
const errorText = document.querySelector('.error-text');
const copyBtn = document.getElementById('copy-btn');
const speakBtn = document.getElementById('speak-btn');
const settingsBtn = document.getElementById('settings-btn');
const openWebBtn = document.getElementById('open-web-btn');

let currentTranslation = '';

/**
 * 初始化
 */
async function init() {
    await api.init();
    await loadLanguages();
    await loadConfig();
    bindEvents();
}

/**
 * 加载语言列表
 */
async function loadLanguages() {
    try {
        const result = await api.getLanguages();
        if (result.success && result.languages) {
            const languages = result.languages;

            // 填充源语言选择器（带国旗）
            sourceLangSelect.innerHTML = '';
            languages.forEach(lang => {
                const option = document.createElement('option');
                option.value = lang.code;
                const displayName = lang.displayName || `${lang.flag || '🌐'} ${lang.native_name || lang.name}`;
                option.textContent = displayName;
                option.dataset.flag = lang.flag || '🌐';
                sourceLangSelect.appendChild(option);
            });

            // 填充目标语言选择器（带国旗）
            targetLangSelect.innerHTML = '';
            languages.filter(lang => lang.code !== 'auto').forEach(lang => {
                const option = document.createElement('option');
                option.value = lang.code;
                const displayName = lang.displayName || `${lang.flag || '🌐'} ${lang.native_name || lang.name}`;
                option.textContent = displayName;
                option.dataset.flag = lang.flag || '🌐';
                targetLangSelect.appendChild(option);
            });

            // 设置默认值
            const config = await api.getConfig();
            if (config.defaultSourceLang) {
                sourceLangSelect.value = config.defaultSourceLang;
            }
            if (config.defaultTargetLang) {
                targetLangSelect.value = config.defaultTargetLang;
            }
        }
    } catch (error) {
        console.error('加载语言列表失败:', error);
    }
}

/**
 * 加载配置
 */
async function loadConfig() {
    const config = await api.getConfig();
    if (config.useAI) {
        useAICheckbox.checked = true;
    }
}

/**
 * 绑定事件
 */
function bindEvents() {
    translateBtn.addEventListener('click', handleTranslate);
    translatePageBtn.addEventListener('click', handleTranslatePage);
    textInput.addEventListener('keydown', (e) => {
        if (e.key === 'Enter' && (e.ctrlKey || e.metaKey)) {
            handleTranslate();
        }
    });
    copyBtn.addEventListener('click', handleCopy);
    speakBtn.addEventListener('click', handleSpeak);
    settingsBtn.addEventListener('click', () => {
        openOptionsPage();
    });
    openWebBtn.addEventListener('click', async (e) => {
        e.preventDefault();
        try {
            const config = await api.getConfig();
            const url = config.apiUrl || 'https://translate.worldhub.world';
            if (typeof chrome !== 'undefined' && chrome.tabs) {
                chrome.tabs.create({ url: url });
            } else if (typeof browser !== 'undefined' && browser.tabs) {
                browser.tabs.create({ url: url });
            }
            window.close();
        } catch (error) {
            console.error('打开网页失败:', error);
            // 使用默认URL
            const defaultUrl = 'https://translate.worldhub.world';
            if (typeof chrome !== 'undefined' && chrome.tabs) {
                chrome.tabs.create({ url: defaultUrl });
            } else if (typeof browser !== 'undefined' && browser.tabs) {
                browser.tabs.create({ url: defaultUrl });
            }
            window.close();
        }
    });
}

/**
 * 处理翻译
 */
async function handleTranslate() {
    const text = textInput.value.trim();
    if (!text) {
        showError('请输入要翻译的文本');
        return;
    }

    // 显示加载状态
    translateBtn.disabled = true;
    translateBtn.textContent = '翻译中...';
    hideError();
    hideResult();

    try {
        const sourceLang = sourceLangSelect.value;
        const targetLang = targetLangSelect.value;
        const useAI = useAICheckbox.checked;

        const result = await api.translateText(text, sourceLang, targetLang, useAI);

        if (result.success) {
            currentTranslation = result.translated_text;
            showResult(result.translated_text);
        } else {
            showError(result.error || '翻译失败');
        }
    } catch (error) {
        console.error('翻译错误:', error);
        showError('翻译失败: ' + error.message);
    } finally {
        translateBtn.disabled = false;
        translateBtn.textContent = '翻译';
    }
}

/**
 * 显示结果
 */
function showResult(text) {
    resultText.textContent = text;
    resultSection.style.display = 'block';
    errorSection.style.display = 'none';
}

/**
 * 隐藏结果
 */
function hideResult() {
    resultSection.style.display = 'none';
}

/**
 * 显示错误
 */
function showError(message) {
    errorText.textContent = message;
    errorSection.style.display = 'block';
    resultSection.style.display = 'none';
}

/**
 * 隐藏错误
 */
function hideError() {
    errorSection.style.display = 'none';
}

/**
 * 处理复制
 */
async function handleCopy() {
    if (!currentTranslation) return;

    try {
        await navigator.clipboard.writeText(currentTranslation);
        showNotification('已复制到剪贴板');
    } catch (error) {
        console.error('复制失败:', error);
        showNotification('复制失败', 'error');
    }
}

/**
 * 处理朗读
 */
async function handleSpeak() {
    if (!currentTranslation) return;

    try {
        const targetLang = targetLangSelect.value;
        const result = await api.request('/api/tts/speak', {
            method: 'POST',
            body: JSON.stringify({
                text: currentTranslation,
                language: targetLang
            })
        });

        if (result.success && result.audio_url) {
            const audio = new Audio(result.audio_url);
            audio.play().catch(error => {
                console.error('播放失败:', error);
                showNotification('播放失败', 'error');
            });
        } else {
            // 使用浏览器TTS
            if ('speechSynthesis' in window) {
                const utterance = new SpeechSynthesisUtterance(currentTranslation);
                utterance.lang = targetLang;
                speechSynthesis.speak(utterance);
            }
        }
    } catch (error) {
        console.error('朗读失败:', error);
        // 使用浏览器TTS
        if ('speechSynthesis' in window) {
            const utterance = new SpeechSynthesisUtterance(currentTranslation);
            utterance.lang = targetLangSelect.value;
            speechSynthesis.speak(utterance);
        }
    }
}

/**
 * 显示通知
 */
function showNotification(message, type = 'success') {
    // 简单的通知实现
    const notification = document.createElement('div');
    notification.className = `notification notification-${type}`;
    notification.textContent = message;
    notification.style.cssText = `
        position: fixed;
        top: 10px;
        right: 10px;
        padding: 8px 16px;
        background: ${type === 'error' ? '#ef4444' : '#10b981'};
        color: white;
        border-radius: 4px;
        font-size: 12px;
        z-index: 10000;
    `;
    document.body.appendChild(notification);

    setTimeout(() => {
        notification.remove();
    }, 2000);
}

/**
 * 打开选项页面
 */
function openOptionsPage() {
    const optionsUrl = typeof chrome !== 'undefined' && chrome.runtime
        ? chrome.runtime.getURL('options.html')
        : browser.runtime.getURL('options.html');

    if (typeof chrome !== 'undefined' && chrome.tabs) {
        chrome.tabs.create({ url: optionsUrl });
    } else if (typeof browser !== 'undefined' && browser.tabs) {
        browser.tabs.create({ url: optionsUrl });
    }
    window.close();
}

/**
 * 翻译当前页面
 */
async function handleTranslatePage() {
    try {
        const sourceLang = sourceLangSelect.value;
        const targetLang = targetLangSelect.value;
        const useAI = useAICheckbox.checked;
        
        // 显示加载状态
        translatePageBtn.disabled = true;
        translatePageBtn.textContent = '翻译中...';
        hideError();
        hideResult();
        
        // 获取当前活动标签页
        const runtime = typeof chrome !== 'undefined' && chrome.runtime 
            ? chrome 
            : browser;
        
        const tabs = await new Promise((resolve) => {
            if (runtime.tabs) {
                runtime.tabs.query({ active: true, currentWindow: true }, resolve);
            } else {
                resolve([]);
            }
        });
        
        if (tabs.length === 0) {
            showError('无法获取当前页面');
            translatePageBtn.disabled = false;
            translatePageBtn.textContent = '🌐 翻译页面';
            return;
        }
        
        const currentTab = tabs[0];
        
        // 检查并注入 content script（如果需要）
        try {
            // 先尝试发送一个ping消息，检查content script是否已加载
            let contentScriptReady = false;
            try {
                const pingResponse = await new Promise((resolve) => {
                    const timeout = setTimeout(() => resolve(null), 1000);
                    runtime.tabs.sendMessage(currentTab.id, { action: 'ping' }, (response) => {
                        clearTimeout(timeout);
                        if (runtime.lastError) {
                            resolve(null);
                        } else {
                            resolve(response);
                        }
                    });
                });
                if (pingResponse) {
                    contentScriptReady = true;
                }
            } catch (e) {
                // Content script未加载
            }
            
            // 如果content script未加载，尝试注入
            if (!contentScriptReady) {
                if (runtime.scripting && runtime.scripting.executeScript) {
                    // Chrome Manifest V3 - 需要同时注入api.js和content.js
                    try {
                        await runtime.scripting.executeScript({
                            target: { tabId: currentTab.id },
                            files: ['api.js', 'content.js']
                        });
                        // 等待content script初始化
                        await new Promise(resolve => setTimeout(resolve, 500));
                    } catch (injectError) {
                        console.log('Content script注入失败:', injectError);
                        // 继续尝试发送消息，可能已经通过manifest注入
                    }
                } else if (runtime.tabs && runtime.tabs.executeScript) {
                    // Chrome Manifest V2 或 Firefox
                    try {
                        await new Promise((resolve) => {
                            runtime.tabs.executeScript(currentTab.id, {
                                file: 'api.js'
                            }, () => {
                                runtime.tabs.executeScript(currentTab.id, {
                                    file: 'content.js'
                                }, () => {
                                    resolve();
                                });
                            });
                        });
                        // 等待content script初始化
                        await new Promise(resolve => setTimeout(resolve, 500));
                    } catch (injectError) {
                        console.log('Content script注入失败:', injectError);
                    }
                }
            }
        } catch (injectError) {
            console.log('Content script检查失败:', injectError);
        }
        
        // 向content script发送翻译页面消息
        try {
            if (runtime.tabs && runtime.tabs.sendMessage) {
                runtime.tabs.sendMessage(currentTab.id, {
                    action: 'translatePage',
                    sourceLang: sourceLang,
                    targetLang: targetLang,
                    useAI: useAI
                }, (response) => {
                    translatePageBtn.disabled = false;
                    translatePageBtn.textContent = '🌐 翻译页面';
                    
                    // 检查错误
                    if (chrome.runtime && chrome.runtime.lastError) {
                        const errorMsg = chrome.runtime.lastError.message;
                        if (errorMsg.includes('Receiving end does not exist') || errorMsg.includes('Could not establish connection')) {
                            showError('翻译页面失败: Content script未加载。请刷新页面后重试。');
                        } else {
                            showError('翻译页面失败: ' + errorMsg);
                        }
                        return;
                    }
                    
                    if (browser && browser.runtime && browser.runtime.lastError) {
                        const errorMsg = browser.runtime.lastError.message;
                        if (errorMsg.includes('Receiving end does not exist') || errorMsg.includes('Could not establish connection')) {
                            showError('翻译页面失败: Content script未加载。请刷新页面后重试。');
                        } else {
                            showError('翻译页面失败: ' + errorMsg);
                        }
                        return;
                    }
                    
                    if (response && response.success) {
                        showNotification('页面翻译已开始', 'success');
                    } else {
                        showError(response?.error || '翻译页面失败');
                    }
                });
            } else {
                throw new Error('tabs.sendMessage不可用');
            }
        } catch (error) {
            translatePageBtn.disabled = false;
            translatePageBtn.textContent = '🌐 翻译页面';
            console.error('发送消息失败:', error);
            showError('翻译页面失败: ' + error.message + '。请刷新页面后重试。');
        }
    } catch (error) {
        console.error('翻译页面错误:', error);
        showError('翻译页面失败: ' + error.message);
        translatePageBtn.disabled = false;
        translatePageBtn.textContent = '🌐 翻译页面';
    }
}

// 初始化
init().catch(console.error);

