/**
 * Microsoft Office Add-in - Taskpane脚本
 */
Office.onReady((info) => {
    if (info.host === Office.HostType.Word || 
        info.host === Office.HostType.Excel || 
        info.host === Office.HostType.PowerPoint) {
        document.addEventListener("DOMContentLoaded", () => {
            init();
        });
    }
});

const api = new TranslationAPI();

// DOM元素
const textInput = document.getElementById('text-input');
const sourceLangSelect = document.getElementById('source-lang');
const targetLangSelect = document.getElementById('target-lang');
const useAICheckbox = document.getElementById('use-ai');
const translateBtn = document.getElementById('translate-btn');
const translateSelectionBtn = document.getElementById('translate-selection-btn');
const resultSection = document.getElementById('result-section');
const resultText = document.getElementById('result-text');
const errorSection = document.getElementById('error-section');
const errorText = document.getElementById('error-text');
const copyBtn = document.getElementById('copy-btn');
const replaceBtn = document.getElementById('replace-btn');
const speakBtn = document.getElementById('speak-btn');

let currentTranslation = '';
let originalText = '';

/**
 * 初始化
 */
async function init() {
    await api.init();
    await loadLanguages();
    await loadConfig();
    bindEvents();
    
    // 尝试获取Office选中的文本
    try {
        const selectedText = await getOfficeSelectedText();
        if (selectedText) {
            textInput.value = selectedText;
            originalText = selectedText;
        }
    } catch (error) {
        console.log('无法获取Office选中文本:', error);
    }
}

/**
 * 获取Office选中的文本
 */
async function getOfficeSelectedText() {
    return new Promise((resolve, reject) => {
        try {
            if (Office.context.platform === Office.PlatformType.PC || 
                Office.context.platform === Office.PlatformType.Mac) {
                Office.context.document.getSelectedDataAsync(
                    Office.CoercionType.Text,
                    (result) => {
                        if (result.status === Office.AsyncResultStatus.Succeeded) {
                            resolve(result.value || '');
                        } else {
                            resolve('');
                        }
                    }
                );
            } else {
                resolve('');
            }
        } catch (error) {
            resolve('');
        }
    });
}

/**
 * 加载语言列表
 */
async function loadLanguages() {
    try {
        const result = await api.getLanguages();
        if (result.success && result.languages) {
            const languages = result.languages;

            // 填充源语言选择器
            sourceLangSelect.innerHTML = '';
            languages.forEach(lang => {
                const option = document.createElement('option');
                option.value = lang.code;
                option.textContent = lang.displayName || `${lang.flag || '🌐'} ${lang.native_name || lang.name}`;
                sourceLangSelect.appendChild(option);
            });

            // 填充目标语言选择器
            targetLangSelect.innerHTML = '';
            languages.filter(lang => lang.code !== 'auto').forEach(lang => {
                const option = document.createElement('option');
                option.value = lang.code;
                option.textContent = lang.displayName || `${lang.flag || '🌐'} ${lang.native_name || lang.name}`;
                targetLangSelect.appendChild(option);
            });

            // 设置默认值
            const config = await api.getConfig();
            if (config.defaultSourceLang) {
                sourceLangSelect.value = config.defaultSourceLang;
            }
            if (config.defaultTargetLang) {
                targetLangSelect.value = config.defaultTargetLang;
            }
        }
    } catch (error) {
        console.error('加载语言列表失败:', error);
    }
}

/**
 * 加载配置
 */
async function loadConfig() {
    const config = await api.getConfig();
    if (config.useAI) {
        useAICheckbox.checked = true;
    }
}

/**
 * 绑定事件
 */
function bindEvents() {
    translateBtn.addEventListener('click', handleTranslate);
    translateSelectionBtn.addEventListener('click', handleTranslateSelection);
    copyBtn.addEventListener('click', handleCopy);
    replaceBtn.addEventListener('click', handleReplace);
    speakBtn.addEventListener('click', handleSpeak);
}

/**
 * 处理翻译
 */
async function handleTranslate() {
    const text = textInput.value.trim();
    if (!text) {
        showError('请输入要翻译的文本');
        return;
    }

    originalText = text;
    translateBtn.disabled = true;
    translateBtn.querySelector('.ms-Button-label').textContent = '翻译中...';
    hideError();
    hideResult();

    try {
        const sourceLang = sourceLangSelect.value;
        const targetLang = targetLangSelect.value;
        const useAI = useAICheckbox.checked;

        const result = await api.translateText(text, sourceLang, targetLang, useAI);

        if (result.success) {
            currentTranslation = result.translated_text;
            showResult(result.translated_text);
        } else {
            showError(result.error || '翻译失败');
        }
    } catch (error) {
        console.error('翻译错误:', error);
        showError('翻译失败: ' + error.message);
    } finally {
        translateBtn.disabled = false;
        translateBtn.querySelector('.ms-Button-label').textContent = '翻译';
    }
}

/**
 * 处理翻译选中文本
 */
async function handleTranslateSelection() {
    try {
        const selectedText = await getOfficeSelectedText();
        if (!selectedText) {
            showError('请先在文档中选中要翻译的文本');
            return;
        }
        
        textInput.value = selectedText;
        originalText = selectedText;
        await handleTranslate();
    } catch (error) {
        console.error('翻译选中文本失败:', error);
        showError('翻译失败: ' + error.message);
    }
}

/**
 * 替换原文
 */
async function handleReplace() {
    if (!currentTranslation) return;

    try {
        Office.context.document.setSelectedDataAsync(
            currentTranslation,
            { coercionType: Office.CoercionType.Text },
            (result) => {
                if (result.status === Office.AsyncResultStatus.Succeeded) {
                    showNotification('已替换原文');
                } else {
                    showNotification('替换失败: ' + result.error.message, 'error');
                }
            }
        );
    } catch (error) {
        console.error('替换失败:', error);
        showNotification('替换失败: ' + error.message, 'error');
    }
}

/**
 * 显示结果
 */
function showResult(text) {
    resultText.textContent = text;
    resultSection.style.display = 'block';
    errorSection.style.display = 'none';
}

/**
 * 隐藏结果
 */
function hideResult() {
    resultSection.style.display = 'none';
}

/**
 * 显示错误
 */
function showError(message) {
    errorText.textContent = message;
    errorSection.style.display = 'block';
    resultSection.style.display = 'none';
}

/**
 * 隐藏错误
 */
function hideError() {
    errorSection.style.display = 'none';
}

/**
 * 处理复制
 */
async function handleCopy() {
    if (!currentTranslation) return;

    try {
        await navigator.clipboard.writeText(currentTranslation);
        showNotification('已复制到剪贴板');
    } catch (error) {
        console.error('复制失败:', error);
        showNotification('复制失败', 'error');
    }
}

/**
 * 处理朗读
 */
async function handleSpeak() {
    if (!currentTranslation) return;

    try {
        const targetLang = targetLangSelect.value;
        const result = await api.request('/api/tts/speak', {
            method: 'POST',
            body: JSON.stringify({
                text: currentTranslation,
                language: targetLang
            })
        });

        if (result.success && result.audio_url) {
            const audio = new Audio(result.audio_url);
            audio.play().catch(error => {
                console.error('播放失败:', error);
                showNotification('播放失败', 'error');
            });
        } else {
            if ('speechSynthesis' in window) {
                const utterance = new SpeechSynthesisUtterance(currentTranslation);
                utterance.lang = targetLang;
                speechSynthesis.speak(utterance);
            }
        }
    } catch (error) {
        console.error('朗读失败:', error);
        if ('speechSynthesis' in window) {
            const utterance = new SpeechSynthesisUtterance(currentTranslation);
            utterance.lang = targetLangSelect.value;
            speechSynthesis.speak(utterance);
        }
    }
}

/**
 * 显示通知
 */
function showNotification(message, type = 'success') {
    // 简单的通知实现
    const notification = document.createElement('div');
    notification.className = `notification notification-${type}`;
    notification.textContent = message;
    notification.style.cssText = `
        position: fixed;
        top: 10px;
        right: 10px;
        padding: 8px 16px;
        background: ${type === 'error' ? '#ef4444' : '#10b981'};
        color: white;
        border-radius: 4px;
        font-size: 12px;
        z-index: 10000;
    `;
    document.body.appendChild(notification);

    setTimeout(() => {
        notification.remove();
    }, 2000);
}

