/**
 * WPS Office 翻译插件 - Popup脚本
 */
const api = new TranslationAPI();

// DOM元素
const textInput = document.getElementById('text-input');
const sourceLangSelect = document.getElementById('source-lang');
const targetLangSelect = document.getElementById('target-lang');
const useAICheckbox = document.getElementById('use-ai');
const translateBtn = document.getElementById('translate-btn');
const translateSelectionBtn = document.getElementById('translate-selection-btn');
const resultSection = document.getElementById('result-section');
const resultText = document.getElementById('result-text');
const errorSection = document.getElementById('error-section');
const errorText = document.querySelector('.error-text');
const copyBtn = document.getElementById('copy-btn');
const speakBtn = document.getElementById('speak-btn');
const replaceBtn = document.getElementById('replace-btn');
const settingsBtn = document.getElementById('settings-btn');

let currentTranslation = '';
let originalText = '';

/**
 * 初始化
 */
async function init() {
    await api.init();
    await loadLanguages();
    await loadConfig();
    bindEvents();
    
    // 尝试获取WPS Office选中的文本
    try {
        const selectedText = await getWPSSelectedText();
        if (selectedText) {
            textInput.value = selectedText;
            originalText = selectedText;
        }
    } catch (error) {
        console.log('无法获取WPS选中文本:', error);
    }
}

/**
 * 获取WPS Office选中的文本
 */
async function getWPSSelectedText() {
    return new Promise((resolve) => {
        // WPS Office API调用
        if (typeof WPS !== 'undefined' && WPS.Application) {
            try {
                const app = WPS.Application;
                const selection = app.Selection;
                if (selection && selection.Text) {
                    resolve(selection.Text);
                } else {
                    resolve('');
                }
            } catch (error) {
                console.error('获取WPS选中文本失败:', error);
                resolve('');
            }
        } else {
            // 降级：尝试从剪贴板获取
            navigator.clipboard.readText().then(resolve).catch(() => resolve(''));
        }
    });
}

/**
 * 加载语言列表
 */
async function loadLanguages() {
    try {
        const result = await api.getLanguages();
        if (result.success && result.languages) {
            const languages = result.languages;

            // 填充源语言选择器
            sourceLangSelect.innerHTML = '';
            languages.forEach(lang => {
                const option = document.createElement('option');
                option.value = lang.code;
                option.textContent = lang.displayName || `${lang.flag || '🌐'} ${lang.native_name || lang.name}`;
                sourceLangSelect.appendChild(option);
            });

            // 填充目标语言选择器
            targetLangSelect.innerHTML = '';
            languages.filter(lang => lang.code !== 'auto').forEach(lang => {
                const option = document.createElement('option');
                option.value = lang.code;
                option.textContent = lang.displayName || `${lang.flag || '🌐'} ${lang.native_name || lang.name}`;
                targetLangSelect.appendChild(option);
            });

            // 设置默认值
            const config = await api.getConfig();
            if (config.defaultSourceLang) {
                sourceLangSelect.value = config.defaultSourceLang;
            }
            if (config.defaultTargetLang) {
                targetLangSelect.value = config.defaultTargetLang;
            }
        }
    } catch (error) {
        console.error('加载语言列表失败:', error);
    }
}

/**
 * 加载配置
 */
async function loadConfig() {
    const config = await api.getConfig();
    if (config.useAI) {
        useAICheckbox.checked = true;
    }
}

/**
 * 绑定事件
 */
function bindEvents() {
    translateBtn.addEventListener('click', handleTranslate);
    translateSelectionBtn.addEventListener('click', handleTranslateSelection);
    textInput.addEventListener('keydown', (e) => {
        if (e.key === 'Enter' && (e.ctrlKey || e.metaKey)) {
            handleTranslate();
        }
    });
    copyBtn.addEventListener('click', handleCopy);
    speakBtn.addEventListener('click', handleSpeak);
    replaceBtn.addEventListener('click', handleReplace);
    settingsBtn.addEventListener('click', () => {
        showSettings();
    });
}

/**
 * 处理翻译
 */
async function handleTranslate() {
    const text = textInput.value.trim();
    if (!text) {
        showError('请输入要翻译的文本');
        return;
    }

    originalText = text;
    translateBtn.disabled = true;
    translateBtn.textContent = '翻译中...';
    hideError();
    hideResult();

    try {
        const sourceLang = sourceLangSelect.value;
        const targetLang = targetLangSelect.value;
        const useAI = useAICheckbox.checked;

        const result = await api.translateText(text, sourceLang, targetLang, useAI);

        if (result.success) {
            currentTranslation = result.translated_text;
            showResult(result.translated_text);
        } else {
            showError(result.error || '翻译失败');
        }
    } catch (error) {
        console.error('翻译错误:', error);
        showError('翻译失败: ' + error.message);
    } finally {
        translateBtn.disabled = false;
        translateBtn.textContent = '翻译';
    }
}

/**
 * 处理翻译选中文本
 */
async function handleTranslateSelection() {
    try {
        const selectedText = await getWPSSelectedText();
        if (!selectedText) {
            showError('请先在文档中选中要翻译的文本');
            return;
        }
        
        textInput.value = selectedText;
        originalText = selectedText;
        await handleTranslate();
    } catch (error) {
        console.error('翻译选中文本失败:', error);
        showError('翻译失败: ' + error.message);
    }
}

/**
 * 替换原文
 */
async function handleReplace() {
    if (!currentTranslation) return;

    try {
        // 使用WPS Office API替换选中文本
        if (typeof WPS !== 'undefined' && WPS.Application) {
            const app = WPS.Application;
            const selection = app.Selection;
            if (selection) {
                selection.Text = currentTranslation;
                showNotification('已替换原文');
            } else {
                // 降级：复制到剪贴板
                await navigator.clipboard.writeText(currentTranslation);
                showNotification('已复制到剪贴板，请手动粘贴');
            }
        } else {
            // 降级：复制到剪贴板
            await navigator.clipboard.writeText(currentTranslation);
            showNotification('已复制到剪贴板，请手动粘贴');
        }
    } catch (error) {
        console.error('替换失败:', error);
        showNotification('替换失败: ' + error.message, 'error');
    }
}

/**
 * 显示结果
 */
function showResult(text) {
    resultText.textContent = text;
    resultSection.style.display = 'block';
    errorSection.style.display = 'none';
}

/**
 * 隐藏结果
 */
function hideResult() {
    resultSection.style.display = 'none';
}

/**
 * 显示错误
 */
function showError(message) {
    errorText.textContent = message;
    errorSection.style.display = 'block';
    resultSection.style.display = 'none';
}

/**
 * 隐藏错误
 */
function hideError() {
    errorSection.style.display = 'none';
}

/**
 * 处理复制
 */
async function handleCopy() {
    if (!currentTranslation) return;

    try {
        await navigator.clipboard.writeText(currentTranslation);
        showNotification('已复制到剪贴板');
    } catch (error) {
        console.error('复制失败:', error);
        showNotification('复制失败', 'error');
    }
}

/**
 * 处理朗读
 */
async function handleSpeak() {
    if (!currentTranslation) return;

    try {
        const targetLang = targetLangSelect.value;
        const result = await api.request('/api/tts/speak', {
            method: 'POST',
            body: JSON.stringify({
                text: currentTranslation,
                language: targetLang
            })
        });

        if (result.success && result.audio_url) {
            const audio = new Audio(result.audio_url);
            audio.play().catch(error => {
                console.error('播放失败:', error);
                showNotification('播放失败', 'error');
            });
        } else {
            // 使用浏览器TTS
            if ('speechSynthesis' in window) {
                const utterance = new SpeechSynthesisUtterance(currentTranslation);
                utterance.lang = targetLang;
                speechSynthesis.speak(utterance);
            }
        }
    } catch (error) {
        console.error('朗读失败:', error);
        if ('speechSynthesis' in window) {
            const utterance = new SpeechSynthesisUtterance(currentTranslation);
            utterance.lang = targetLangSelect.value;
            speechSynthesis.speak(utterance);
        }
    }
}

/**
 * 显示设置
 */
async function showSettings() {
    const config = await api.getConfig();
    const apiUrl = prompt('API地址:', config.apiUrl);
    if (apiUrl) {
        await api.saveConfig({ ...config, apiUrl });
        await api.init();
        showNotification('设置已保存');
    }
}

/**
 * 显示通知
 */
function showNotification(message, type = 'success') {
    const notification = document.createElement('div');
    notification.className = `notification notification-${type}`;
    notification.textContent = message;
    notification.style.cssText = `
        position: fixed;
        top: 10px;
        right: 10px;
        padding: 8px 16px;
        background: ${type === 'error' ? '#ef4444' : '#10b981'};
        color: white;
        border-radius: 4px;
        font-size: 12px;
        z-index: 10000;
    `;
    document.body.appendChild(notification);

    setTimeout(() => {
        notification.remove();
    }, 2000);
}

// 初始化
init().catch(console.error);

